<?php
/**
 * Template Name: Job Applicants Table
 * Description: Shows a table of applicants for a given job (job_id passed in URL). Click "View" to open full profile modal.
 */

get_header();
global $wpdb;

// Ensure sessions available
if (!session_id()) session_start();

// Check farm owner login
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'farm') {
    echo '<div class="container mt-5"><div class="alert alert-danger">You must be logged in as a farm owner to view this page.</div></div>';
    get_footer();
    exit;
}

$farm_user_id = intval($_SESSION['user_id']);
$job_id = intval($_GET['job_id'] ?? 0);

if ($job_id <= 0) {
    echo '<div class="container mt-5"><div class="alert alert-warning">Invalid job ID.</div></div>';
    get_footer();
    exit;
}

// Verify job belongs to this farm owner
$job = $wpdb->get_row($wpdb->prepare("SELECT * FROM farm_jobs WHERE id = %d AND farmer_id = %d", $job_id, $farm_user_id));
if (!$job) {
    echo '<div class="container mt-5"><div class="alert alert-danger">Job not found or access denied.</div></div>';
    get_footer();
    exit;
}

// Fetch applicants (basic info) joined with applications table
$applicants = $wpdb->get_results($wpdb->prepare("
    SELECT ui.id as user_info_id, ui.username, ui.email, ui.contact,
           up.birthdate, up.city, up.bio, up.skills, up.image,
           ja.applied_on
    FROM job_applications ja
    INNER JOIN user_info ui ON ja.user_id = ui.id
    INNER JOIN user_profiles up ON up.user_id = ui.id
    WHERE ja.job_id = %d
    ORDER BY ja.applied_on DESC
", $job_id));

?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Applicants for <?php echo esc_html($job->role); ?></title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- Bootstrap CSS -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body { font-family: 'League Spartan', sans-serif; background:#f9f9f9; }
    .page-wrap { max-width: 1000px; margin: 40px auto; background:#fff; padding:25px; border-radius:12px; box-shadow:0 6px 24px rgba(0,0,0,0.06); }
    h2.title { color:#fc3c68; margin-bottom:18px; font-weight:700; }
    table thead { background:#fc3c68; color:#fff; }
    .btn-view { background:#3173BD; color:#fff; border-radius:6px; }
    .profile-img { width:90px; height:90px; object-fit:cover; border-radius:50%; border:3px solid #fc3c68; }
    .hidden-profile { display:none; }
    .user-name{ color: #fc3c68;    font-weight: 700;    margin-bottom: 10px;}
    .section-title {

    color: #333;
    font-weight: 600;
    font-size:12px;
}
  </style>
</head>
<body>

<div class="container page-wrap">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <div>
      <h2 class="title">Applicants for: <?php echo esc_html($job->role); ?></h2>
      <p class="mb-0"><strong>Job ID:</strong> <?php echo esc_html($job->id); ?> &nbsp; | &nbsp; <strong>City:</strong> <?php echo esc_html($job->city ?? 'N/A'); ?></p>
    </div>
    <div>
      <a href="<?php echo esc_url(home_url('/farm-dashboard')); ?>" class="btn btn-outline-secondary">← Back to Jobs</a>
    </div>
  </div>

  <?php if (empty($applicants)): ?>
    <div class="alert alert-warning">No applicants have applied for this job yet.</div>
  <?php else: ?>
    <div class="table-responsive">
      <table class="table table-striped table-bordered align-middle">
        <thead>
          <tr>
            <th>#</th>
            <th>Name</th>
            <th>Experience (latest)</th>
            <th>Skills</th>
            <th>Applied On</th>
            <th>Action</th>
          </tr>
        </thead>
        <tbody>
          <?php
          $i = 1;
          foreach ($applicants as $applicant):
              $user_id = $applicant->user_info_id;

              // Latest experience summary
              $latest_exp = $wpdb->get_row($wpdb->prepare(
                  "SELECT * FROM user_experience WHERE user_id = %d ORDER BY start_date DESC LIMIT 1",
                  $user_id
              ));
              $latest_exp_text = $latest_exp ? esc_html($latest_exp->job_designation) . ' at ' . esc_html($latest_exp->company_name) : 'N/A';

              // Short skills
              $skills_short = !empty($applicant->skills) ? wp_trim_words($applicant->skills, 10, '...') : 'N/A';

              // Profile image
              $img_base = site_url('/wp-content/themes/onum/page-templates/uploads/');
              $img_url = !empty($applicant->image) ? esc_url($img_base . $applicant->image) : esc_url($img_base . 'default.png');
          ?>
            <tr>
              <td><?php echo $i++; ?></td>
              <td><?php echo esc_html($applicant->username); ?></td>
              <td><?php echo $latest_exp_text; ?></td>
              <td><?php echo esc_html($skills_short); ?></td>
              <td><?php echo esc_html($applicant->applied_on); ?></td>
              <td>
                <button class="btn btn-sm btn-view" data-applicant-id="<?php echo esc_attr($user_id); ?>">View</button>
              </td>
            </tr>

            <!-- Hidden full profile -->
            <div id="profile-html-<?php echo esc_attr($user_id); ?>" class="hidden-profile">
              <div style="padding:10px;">
                <div style="display:flex; gap:15px; align-items:center;">
                  <div>
                    <img src="<?php echo $img_url; ?>" alt="Profile" class="profile-img">
                  </div>
                  <div>
                    <h4 class="user-name" style="margin:0;"><?php echo esc_html($applicant->username); ?></h4>
                    <p style="margin:0;"><strong>Email:</strong> <?php echo esc_html($applicant->email); ?></p>
                    <p style="margin:0;"><strong>Contact:</strong> <?php echo esc_html($applicant->contact); ?></p>
                    <p style="margin:0;"><strong>City:</strong> <?php echo esc_html($applicant->city); ?></p>
                    <p style="margin:0;"><strong>Applied On:</strong> <?php echo esc_html($applicant->applied_on); ?></p>
                  </div>
                </div>

                <hr>

                <div style="margin-top:8px;">
                  <h6 class="section-title">Bio</h6>
                  <p><?php echo nl2br(esc_html($applicant->bio)); ?></p>
                </div>

                <div style="margin-top:8px;">
                  <h6 class="section-title">Skills</h6>
                  <p><?php echo nl2br(esc_html($applicant->skills)); ?></p>
                </div>

                <?php
                // Education
                $edus = $wpdb->get_results($wpdb->prepare("SELECT * FROM user_education WHERE user_id = %d ORDER BY completion_date DESC", $user_id));
                if (!empty($edus)):
                ?>
                  <div style="margin-top:10px;">
                    <h6 class="section-title">Education</h6>
                    <ul>
                      <?php foreach ($edus as $ed): ?>
                        <li><strong><?php echo esc_html($ed->course); ?></strong> — <?php echo esc_html($ed->institute); ?> (<?php echo esc_html($ed->completion_date); ?>)</li>
                      <?php endforeach; ?>
                    </ul>
                  </div>
                <?php endif; ?>

                <?php
                // Experience
                $exps = $wpdb->get_results($wpdb->prepare("SELECT * FROM user_experience WHERE user_id = %d ORDER BY start_date DESC", $user_id));
                if (!empty($exps)):
                ?>
                  <div style="margin-top:10px;">
                    <h6 class="section-title">Experience</h6>
                    <?php foreach ($exps as $ex):
                        $end = $ex->end_date ? esc_html($ex->end_date) : 'Present';
                    ?>
                      <div style="margin-bottom:10px;">
                        <span><?php echo esc_html($ex->job_designation); ?></span> -
                        <span><?php echo esc_html($ex->company_name); ?></span><br>
                        <small>Period: <?php echo esc_html($ex->start_date); ?> to <?php echo $end; ?></small>
                        <p>Responsibilities: <?php echo nl2br(esc_html($ex->work_responsibilities)); ?></p>
                      </div>
                    <?php endforeach; ?>
                  </div>
                <?php endif; ?>

                <div style="margin-top:12px;">
                  <a href="mailto:<?php echo esc_html($applicant->email); ?>" class="btn btn-sm btn-outline-secondary">Email Applicant</a>
                </div>

              </div>
            </div>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  <?php endif; ?>
</div>

<!-- Modal -->
<div class="modal fade" id="applicantModal" tabindex="-1" aria-labelledby="applicantModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="applicantModalLabel">Applicant Details</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body" id="applicantModalBody">
        <div class="text-center">Loading...</div>
      </div>
      <div class="modal-footer">
        <button class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
      </div>
    </div>
  </div>
</div>

<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
(function(){
  const modalEl = document.getElementById('applicantModal');
  const modal = new bootstrap.Modal(modalEl);
  const modalBody = document.getElementById('applicantModalBody');

  document.querySelectorAll('button[data-applicant-id]').forEach(btn => {
    btn.addEventListener('click', function(){
      const id = this.getAttribute('data-applicant-id');
      const hidden = document.getElementById('profile-html-' + id);
      if (hidden) {
        modalBody.innerHTML = hidden.innerHTML;
        modal.show();
      } else {
        modalBody.innerHTML = '<div class="alert alert-danger">Applicant details not found.</div>';
        modal.show();
      }
    });
  });
})();
</script>

</body>
</html>

<?php get_footer(); ?>
